#!/bin/sh

#
# Bitdefender (c) 2024
#

#
# Configuration script for managing network capture rules.
# Port 8887 should be open.
# Network capture will not be enabled alongside firewalld or ufw.
# Edit this file at own risk. This file will be overwritten on product update.
#

set -e

LC_ALL=C
export LC_ALL

PORT=443
COMPANY_NAME="Bitdefender"
COMPANY_NAME=$(printf "%s" "$COMPANY_NAME" | sed 's/ //g')
COMPANY_NAME=$(printf "%s" "$COMPANY_NAME" | head -c 18)

CHAIN_OUTPUT="$COMPANY_NAME"-"$PORT"-out

if iptables -t raw -L --wait 20 2>/dev/null 1>&2; then
	WAIT_ARG="--wait 5"
else
	iptables -t raw -L --wait 2>/dev/null 1>&2 && WAIT_ARG="--wait"
fi

#
# Naming convention: {TABLE_NAME}_{OPERATION}_{ORIGINAL_CHAIN}_{...}
# TABLE_NAME: F - filter, N - nat, R - raw
# OPERATION: A - append, I - insert
# ORIGINAL_CHAIN: O - output, P - prerouting, I - input
#

F_I_O_ACCEPT="$CHAIN_OUTPUT -p tcp -d 127.0.0.1 --dport 8887 -j ACCEPT -m owner ! --gid-owner $OWNER_GID -m mark ! --mark 0x3887"
F_I_O_CHAIN="OUTPUT -j $CHAIN_OUTPUT"

R_A_O_NFQ="$CHAIN_OUTPUT -j NFQUEUE -p tcp --dport $PORT -m owner ! --gid-owner $OWNER_GID --syn --queue-num 0"
R_A_O_CHAIN="OUTPUT -j $CHAIN_OUTPUT"

N_A_O_CLEARMARK="$CHAIN_OUTPUT -j MARK --set-mark 0"
N_A_O_DNAT="$CHAIN_OUTPUT -p tcp --dport $PORT -m owner ! --gid-owner $OWNER_GID -j DNAT --to 127.0.0.1:8887"
N_A_O_MARK="$CHAIN_OUTPUT -j MARK --set-mark 0x3887"
N_A_O_CHAIN="OUTPUT -j $CHAIN_OUTPUT"

check_firewall()
{
	if firewall-cmd --state 2>/dev/null 1>&2; then
		exit 3
	fi

	status="$(ufw status 2>/dev/null | grep ": active" || true)"
	if [ -n "$status" ]; then
		exit 3
	fi

	return 0
}

create_chains()
{
	iptables -t filter -L $CHAIN_OUTPUT $WAIT_ARG 2>/dev/null 1>&2 || iptables -t filter -N $CHAIN_OUTPUT $WAIT_ARG

	iptables -t raw -L $CHAIN_OUTPUT $WAIT_ARG 2>/dev/null 1>&2 || iptables -t raw -N $CHAIN_OUTPUT $WAIT_ARG

	iptables -t nat -L $CHAIN_OUTPUT $WAIT_ARG 2>/dev/null 1>&2 || iptables -t nat -N $CHAIN_OUTPUT $WAIT_ARG

	return 0
}

add_rules()
{
	iptables -t filter -C $F_I_O_ACCEPT $WAIT_ARG 2>/dev/null || iptables -t filter -I $F_I_O_ACCEPT $WAIT_ARG

	iptables -t raw -C $R_A_O_NFQ $WAIT_ARG 2>/dev/null || iptables -t raw -A $R_A_O_NFQ $WAIT_ARG

	iptables -t nat -C $N_A_O_CLEARMARK $WAIT_ARG 2>/dev/null || iptables -t nat -A $N_A_O_CLEARMARK $WAIT_ARG

	iptables -t nat -C $N_A_O_DNAT $WAIT_ARG 2>/dev/null || iptables -t nat -A $N_A_O_DNAT $WAIT_ARG

	iptables -t nat -C $N_A_O_MARK $WAIT_ARG 2>/dev/null || iptables -t nat -A $N_A_O_MARK $WAIT_ARG

	return 0
}

insert_chains()
{
	iptables -t filter -C $F_I_O_CHAIN $WAIT_ARG 2>/dev/null || iptables -t filter -I $F_I_O_CHAIN $WAIT_ARG

	iptables -t raw -C $R_A_O_CHAIN $WAIT_ARG 2>/dev/null || iptables -t raw -A $R_A_O_CHAIN $WAIT_ARG

	iptables -t nat -C $N_A_O_CHAIN $WAIT_ARG 2>/dev/null || iptables -t nat -A $N_A_O_CHAIN $WAIT_ARG

	return 0
}

remove_chains()
{
	iptables -t filter -C $F_I_O_CHAIN $WAIT_ARG 2>/dev/null && iptables -t filter -D $F_I_O_CHAIN $WAIT_ARG

	iptables -t raw -C $R_A_O_CHAIN $WAIT_ARG 2>/dev/null && iptables -t raw -D $R_A_O_CHAIN $WAIT_ARG

	iptables -t nat -C $N_A_O_CHAIN $WAIT_ARG 2>/dev/null && iptables -t nat -D $N_A_O_CHAIN $WAIT_ARG

	return 0
}

remove_rules()
{
	iptables -t filter -C $F_I_O_ACCEPT $WAIT_ARG 2>/dev/null && iptables -t filter -D $F_I_O_ACCEPT $WAIT_ARG

	iptables -t raw -C $R_A_O_NFQ $WAIT_ARG 2>/dev/null && iptables -t raw -D $R_A_O_NFQ $WAIT_ARG

	iptables -t nat -C $N_A_O_CLEARMARK $WAIT_ARG 2>/dev/null && iptables -t nat -D $N_A_O_CLEARMARK $WAIT_ARG

	iptables -t nat -C $N_A_O_DNAT $WAIT_ARG 2>/dev/null && iptables -t nat -D $N_A_O_DNAT $WAIT_ARG

	iptables -t nat -C $N_A_O_MARK $WAIT_ARG 2>/dev/null && iptables -t nat -D $N_A_O_MARK $WAIT_ARG

	return 0
}

delete_chains()
{
	iptables -t filter -L $CHAIN_OUTPUT $WAIT_ARG 2>/dev/null 1>&2 && iptables -t filter -X $CHAIN_OUTPUT $WAIT_ARG

	iptables -t raw -L $CHAIN_OUTPUT $WAIT_ARG 2>/dev/null 1>&2 && iptables -t raw -X $CHAIN_OUTPUT $WAIT_ARG

	iptables -t nat -L $CHAIN_OUTPUT $WAIT_ARG 2>/dev/null 1>&2 && iptables -t nat -X $CHAIN_OUTPUT $WAIT_ARG

	return 0
}

check_inserted_chains()
{
	if [ "3" != "$(iptables -t filter -L OUTPUT $WAIT_ARG | grep -no "$COMPANY_NAME" | head -1 | cut -d : -f1)" ]; then
		exit 1
	fi

	exit 0
}

case "$1" in
"start")
	check_firewall

	create_chains

	add_rules

	insert_chains
	;;
"stop")
	remove_chains

	remove_rules

	delete_chains
	;;
"check")
	check_inserted_chains
	;;
esac
